/*-----------------------------------------------------------------------*/
/* Low level disk I/O module skeleton for FatFs     (C)ChaN, 2014        */
/*-----------------------------------------------------------------------*/
/* If a working storage control module is available, it should be        */
/* attached to the FatFs via a glue function rather than modifying it.   */
/* This is an example of glue functions to attach various exsisting      */
/* storage control modules to the FatFs module with a defined API.       */
/*-----------------------------------------------------------------------*/

#include "diskio.h"		/* FatFs lower layer API */
#include "stm324xg_eval_sdio_sd.h"		/* Example: Header file of existing MMC/SDC contorl module */

#include "usb_conf.h"
#include "usbh_msc_core.h"

/*----------------------------------varialbe-----------------------------*/
static volatile DSTATUS Stat = STA_NOINIT;	/* Disk status */

extern USB_OTG_CORE_HANDLE          USB_OTG_Core;
extern USBH_HOST                     USB_Host;


/*-----------------------------------------------------------------------*/
/* Get Drive Status                                                      */
/*-----------------------------------------------------------------------*/
unsigned long int get_fattime(void)
{
	return 0;
}


DSTATUS disk_status (
	BYTE pdrv		/* Physical drive nmuber to identify the drive */
)
{
	if(pdrv == 0){
		return RES_OK;
	}else if(pdrv == 1){
		return Stat;
	}
}



/*-----------------------------------------------------------------------*/
/* Inidialize a Drive                                                    */
/*-----------------------------------------------------------------------*/

DSTATUS disk_initialize (
	BYTE pdrv				/* Physical drive nmuber to identify the drive */
)
{
	if(pdrv == 0){
		return RES_OK;
	}else if(pdrv == 1){
	  if(HCD_IsDeviceConnected(&USB_OTG_Core))
		{  
			Stat &= ~STA_NOINIT;
		}
		
		return Stat;		
	}
}



/*-----------------------------------------------------------------------*/
/* Read Sector(s)                                                        */
/*-----------------------------------------------------------------------*/

DRESULT disk_read (
	BYTE pdrv,		/* Physical drive nmuber to identify the drive */
	BYTE *buff,		/* Data buffer to store read data */
	DWORD sector,	/* Sector address in LBA */
	UINT count		/* Number of sectors to read */
)
{
	if(pdrv == 0){
		if( SD_ReadMultiBlocks (buff, 
														sector * 512, 
														512,
														count) != 0)
		{
			return RES_ERROR;
		}
		SD_WaitReadOperation();
		while (SD_GetStatus() != SD_TRANSFER_OK);	

		return RES_OK;
	}else if(pdrv == 1){
		BYTE status = USBH_MSC_OK;
		
		if (Stat & STA_NOINIT) return RES_NOTRDY;
		
		
		if(HCD_IsDeviceConnected(&USB_OTG_Core))
		{  
			
			do
			{
				status = USBH_MSC_Read10(&USB_OTG_Core, buff,sector,512 * count);
				USBH_MSC_HandleBOTXfer(&USB_OTG_Core ,&USB_Host);
				
				if(!HCD_IsDeviceConnected(&USB_OTG_Core))
				{ 
					return RES_ERROR;
				}      
			}
			while(status == USBH_MSC_BUSY );
		}
		
		if(status == USBH_MSC_OK)
			return RES_OK;
		return RES_ERROR;
	}
}



/*-----------------------------------------------------------------------*/
/* Write Sector(s)                                                       */
/*-----------------------------------------------------------------------*/

DRESULT disk_write (
	BYTE pdrv,			/* Physical drive nmuber to identify the drive */
	const BYTE *buff,	/* Data to be written */
	DWORD sector,		/* Sector address in LBA */
	UINT count			/* Number of sectors to write */
)
{
	if(pdrv == 0){
		if( SD_WriteMultiBlocks ((unsigned char *)buff, 
														 sector * 512, 
														 512,
														 count) != 0)
		{
			return RES_ERROR;
		} 
		SD_WaitWriteOperation();
		while (SD_GetStatus() != SD_TRANSFER_OK);
		return RES_OK;
	}else if(pdrv == 1){
		BYTE status = USBH_MSC_OK;
		if (Stat & STA_NOINIT) return RES_NOTRDY;
		if (Stat & STA_PROTECT) return RES_WRPRT;
		
		
		if(HCD_IsDeviceConnected(&USB_OTG_Core))
		{  
			do
			{
				status = USBH_MSC_Write10(&USB_OTG_Core,(BYTE*)buff,sector,512 * count);
				USBH_MSC_HandleBOTXfer(&USB_OTG_Core, &USB_Host);
				
				if(!HCD_IsDeviceConnected(&USB_OTG_Core))
				{ 
					return RES_ERROR;
				}
			}
			
			while(status == USBH_MSC_BUSY );
			
		}
		
		if(status == USBH_MSC_OK)
			return RES_OK;
		return RES_ERROR;		
	}
}


/*-----------------------------------------------------------------------*/
/* Miscellaneous Functions                                               */
/*-----------------------------------------------------------------------*/
DRESULT disk_ioctl (
	BYTE pdrv,		/* Physical drive nmuber (0..) */
	BYTE cmd,		/* Control code */
	void *buff		/* Buffer to send/receive control data */
)
{
	if(pdrv == 0){
		return RES_OK;
	}else if(pdrv == 1){
		DRESULT res = RES_OK;
		
		res = RES_ERROR;
		
		if (Stat & STA_NOINIT) return RES_NOTRDY;
		
		switch (cmd) {
		case CTRL_SYNC :		/* Make sure that no pending write process */
			
			res = RES_OK;
			break;
			
		case GET_SECTOR_COUNT :	/* Get number of sectors on the disk (DWORD) */
			
			*(DWORD*)buff = (DWORD) USBH_MSC_Param.MSCapacity;
			res = RES_OK;
			break;
			
		case GET_SECTOR_SIZE :	/* Get R/W sector size (WORD) */
			*(WORD*)buff = 512;
			res = RES_OK;
			break;
			
		case GET_BLOCK_SIZE :	/* Get erase block size in unit of sector (DWORD) */
			
			*(DWORD*)buff = 512;
			
			break;			
		default:
			res = RES_PARERR;
		}	
		return res;		
	}
}

